/*
 * Copyright (c) 2019-2022 Amazon.com, Inc. or its affiliates.  All rights
 * reserved.
 * Portions Copyright (C) 2018 Sigmastar Technology Corp
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
/*  A network driver that communicates over SDIO, and only
 *  supports "Packet Sockets" family.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/init.h>
#include <linux/moduleparam.h>
#include <linux/rtnetlink.h>
#include <net/rtnetlink.h>
#include <net/sock.h>
#include <linux/u64_stats_sync.h>

#include "amazon_net.h"

typedef int (*netif_rx_fnc)(struct sk_buff *skb);

struct pcpu_dstats {
	u64			tx_packets;
	u64			tx_bytes;
	u64			rx_packets;
	u64			rx_bytes;
	struct u64_stats_sync	syncp;
};

static const struct amazon_net_ops *backends[] = {
		&loop_ops,
		&amazon_sdio_ops
};

/******************************************************************************
 * net_device_ops functions
 *****************************************************************************/

static int amazon_net_dev_init(struct net_device *dev)
{
	dev->dstats = netdev_alloc_pcpu_stats(struct pcpu_dstats);
	if (!dev->dstats)
		return -ENOMEM;

	return 0;
}

static void amazon_net_dev_uninit(struct net_device *dev)
{
	free_percpu(dev->dstats);
}

static netdev_tx_t amazon_net_netdev_xmit(struct sk_buff *skb,
					  struct net_device *dev)
{
	int ret;
	struct amazon_net_priv *priv;
	struct pcpu_dstats *dstats;
	int len = skb->len;

	if (!dev->parent && skb->sk->sk_family != PF_PACKET) {
		netdev_err(dev,
			   "%s: Network adapter only accepts PF_PACKET family addresses\n",
			   __func__);
		return NETDEV_TX_BUSY;
	}

	dstats = this_cpu_ptr(dev->dstats);
	u64_stats_update_begin(&dstats->syncp);
	dstats->tx_packets++;
	dstats->tx_bytes += len;
	u64_stats_update_end(&dstats->syncp);

#if defined(CONFIG_DYNAMIC_DEBUG) || defined(DEBUG)
	{
		char data_str[200];

		escape_string_buffer(data_str, sizeof(data_str),
				     skb->data + 1, skb->len - 1);
		netdev_vdbg(dev, "sending %d byte packet to addr %d : \"%.*s\"\n",
			    skb->len - 1,
			    skb->data[0],
			    skb->len - 1,  data_str);
	}
#endif // CONFIG_DYNAMIC_DEBUG || DEBUG

	priv = (struct amazon_net_priv *)amazon_net_ndev_priv(dev);

	ret = priv->ops->xmit(dev, skb);

	netdev_dbg(dev, "xmit %d bytes: result %d\n",
		   len, ret);

	// dev_kfree_skb(skb);

	return (ret) ? NETDEV_TX_BUSY : NETDEV_TX_OK;
}

static void
amazon_net_get_stats64(struct net_device *dev, struct rtnl_link_stats64 *stats)
{
	int i;

	for_each_possible_cpu(i) {
		const struct pcpu_dstats *dstats;
		u64 tbytes, tpackets, rbytes, rpackets;
		unsigned int start;

		dstats = per_cpu_ptr(dev->dstats, i);
		do {
			start = u64_stats_fetch_begin_irq(&dstats->syncp);
			tbytes = dstats->tx_bytes;
			tpackets = dstats->tx_packets;
			rbytes = dstats->rx_bytes;
			rpackets = dstats->rx_packets;
		} while (u64_stats_fetch_retry_irq(&dstats->syncp, start));
		stats->tx_bytes += tbytes;
		stats->tx_packets += tpackets;
		stats->rx_bytes += rbytes;
		stats->rx_packets += rpackets;
	}
}

static u16 amazon_net_select_queue(struct net_device *dev,
				   struct sk_buff *skb,
				   struct net_device *sb_dev,
				   select_queue_fallback_t fallback)
{
	struct amazon_net_priv *priv =
			(struct amazon_net_priv *)amazon_net_ndev_priv(dev);

	if (!priv || !priv->ops || !priv->ops->select_queue)
		return 0;

	return priv->ops->select_queue(dev, skb, sb_dev, fallback);
}

static int amazon_net_do_ioctl(struct net_device *dev,
			       struct ifreq *ifr,
			       int cmd)
{
	struct amazon_net_priv *priv =
			(struct amazon_net_priv *)amazon_net_ndev_priv(dev);

	if (!priv || !priv->ops || !priv->ops->do_ioctl)
		return -ENODEV;

	return priv->ops->do_ioctl(dev, ifr, cmd);
}

static int amazon_net_flush_sockaddr(struct net_device *dev,
				     unsigned char sll_addr)
{
	struct amazon_net_priv *priv =
			(struct amazon_net_priv *)amazon_net_ndev_priv(dev);

	if (!priv || !priv->ops || !priv->ops->flush_channel)
		return -ENODEV;

	priv->ops->flush_channel(dev, sll_addr);
	return 0;
}

static const struct net_device_ops amazon_net_netdev_ops = {
	.ndo_init		= amazon_net_dev_init,
	.ndo_uninit		= amazon_net_dev_uninit,
	.ndo_start_xmit		= amazon_net_netdev_xmit,
	.ndo_select_queue	= amazon_net_select_queue,
	.ndo_get_stats64	= amazon_net_get_stats64,
	.ndo_do_ioctl		= amazon_net_do_ioctl,
	.ndo_flush_sockaddr	= amazon_net_flush_sockaddr,
};

/******************************************************************************
 * header_ops functions
 *****************************************************************************/

static int amazon_net_header_create(struct sk_buff *skb,
				    struct net_device *dev,
				    unsigned short type,
				    const void *daddr,
				    const void *saddr,
				    unsigned int len)
{
	unsigned char *hdr = skb_push(skb, 4);

	if (!daddr)
		return -EINVAL;

	if (skb->sk->sk_family != PF_PACKET) {
		netdev_err(dev,
			   "%s: Network adapter only accepts PF_PACKET family addresses\n",
			   __func__);
		return -EINVAL;
	}

	hdr[0] = *((unsigned char *)daddr); // Channel number
	hdr[1] = *(((unsigned char *)&len));     // Length
	hdr[2] = *(((unsigned char *)&len) + 1); // Length
	hdr[3] = 0; // reserved

	// Return the header length - af_packet.c uses it as an offset
	return 4;
}

static int
amazon_net_header_parse(const struct sk_buff *skb, unsigned char *haddr)
{
	memset(haddr, 0, 8);
	*haddr = skb->data[0];
	// Return the header length - af_packet.c uses it as an offset
	return 4;
}

static int amazon_net_ip_header_create(struct sk_buff *skb,
				       struct net_device *dev,
				       unsigned short type,
				       const void *daddr,
				       const void *saddr,
				       unsigned int len)
{
	unsigned char *hdr = skb_push(skb, 4);

	hdr[0] = RED_SDIO_NETIF_CH; // Channel number
	hdr[1] = *(((unsigned char *)&len));     // Length
	hdr[2] = *(((unsigned char *)&len) + 1); // Length
	hdr[3] = 0; // reserved

	// Return the header length - af_packet.c uses it as an offset
	return 4;
}

static const struct header_ops amazon_net_header_ops = {
	.create         = amazon_net_header_create,
	.parse          = amazon_net_header_parse,
};

static const struct header_ops amazon_net_ip_header_ops = {
	.create         = amazon_net_ip_header_create,
};

/******************************************************************************
 * ethtool_ops functions
 *****************************************************************************/

static void amazon_net_get_drvinfo(struct net_device *dev,
				   struct ethtool_drvinfo *info)
{
	strlcpy(info->driver, DRV_NAME, sizeof(info->driver));
	strlcpy(info->version, DRV_VERSION, sizeof(info->version));
}

static const struct ethtool_ops amazon_net_ethtool_ops = {
	.get_drvinfo            = amazon_net_get_drvinfo,
};

/******************************************************************************
 * rtnl_link_ops functions
 *****************************************************************************/

static void amazon_net_setup(struct net_device *dev)
{
	ether_setup(dev);

	/* Initialize the device structure. */
	dev->netdev_ops = &amazon_net_netdev_ops;
	dev->ethtool_ops = &amazon_net_ethtool_ops;
	dev->header_ops = &amazon_net_header_ops;
	dev->priv_destructor = free_netdev;

	/* Fill in device structure with ethernet-generic values. */
	dev->flags |= IFF_NOARP;
	dev->flags &= ~IFF_MULTICAST;
	dev->priv_flags |= IFF_LIVE_ADDR_CHANGE; // | IFF_NO_QUEUE;

	dev->features	= NETIF_F_HW_CSUM;
	dev->hard_header_len = 1;

	dev->hw_features |= dev->features;
	dev->hw_enc_features |= dev->features;

	/* 3072 - Same as the receiving DMA buffers on Red */
	dev->max_mtu = 3072;
	dev->mtu = 3072;
	eth_hw_addr_random(dev);
}

static struct rtnl_link_ops amazon_net_link_ops __read_mostly = {
	.kind		= DRV_NAME,
	.setup		= amazon_net_setup,
};

/******************************************************************************
 * Module Init/Exit functions
 *****************************************************************************/

static int __init amazon_net_init_module(void)
{
	int i;
	int err;

	pr_info("%s\n", __func__);

	for (i = 0; i < ARRAY_SIZE(backends); i++)
		if (backends[i]->init())
			return -1;

	rtnl_lock();
	err = __rtnl_link_register(&amazon_net_link_ops);
	rtnl_unlock();

	return err;
}

static void __exit amazon_net_cleanup_module(void)
{
	int i;

	rtnl_lock();
	__rtnl_link_unregister(&amazon_net_link_ops);
	rtnl_unlock();

	for (i = 0; i < ARRAY_SIZE(backends); i++)
		backends[i]->deinit();
}

/******************************************************************************
 * Functions used by SDIO device driver part
 *****************************************************************************/

struct net_device  *amazon_net_ndev_alloc(int sizeof_priv,
					  const char *name_pattern, bool ip)
{
	struct net_device *ndev;

	ndev = alloc_netdev(sizeof_priv,
			    name_pattern,
			    NET_NAME_UNKNOWN,
			    amazon_net_setup);
	if (!ndev)
		return NULL;

	ndev->addr_len = 1;

	if (ip)
		ndev->header_ops = &amazon_net_ip_header_ops;

	return ndev;
}

void amazon_net_ndev_free(struct net_device  *ndev)
{
	if (ndev)
		free_netdev(ndev);
}

int amazon_net_ndev_register(struct net_device	*ndev,
			     const struct amazon_net_ops *ops)
{
	struct amazon_net_priv *priv;

	if (!ndev || !ops)
		return -EINVAL;

	priv = (struct amazon_net_priv *)amazon_net_ndev_priv(ndev);
	ndev->rtnl_link_ops = &amazon_net_link_ops;
	priv->ops = ops;

	return register_netdevice(ndev);
}

void amazon_net_ndev_unregister(struct net_device  *ndev)
{
	if (ndev)
		unregister_netdev(ndev);
}

struct net_device  *amazon_net_ndev_create(int sizeof_priv,
					   const struct amazon_net_ops *ops,
					   const char *name_pattern)
{
	int ret = -1;
	struct net_device *dev_amazon_net = NULL;

	dev_amazon_net = amazon_net_ndev_alloc(sizeof_priv,
					       name_pattern, false);
	if (!dev_amazon_net)
		return NULL;

	rtnl_lock();
	ret = amazon_net_ndev_register(dev_amazon_net, ops);
	rtnl_unlock();
	if (ret < 0)
		goto exit;

	return dev_amazon_net;

exit:

	amazon_net_ndev_free(dev_amazon_net);
	return NULL;
}

void amazon_net_ndev_delete(struct net_device  *ndev)
{
	amazon_net_ndev_unregister(ndev);
	amazon_net_ndev_free(ndev);
}

int amazon_net_receive(struct net_device *ndev, struct sk_buff *skb, bool ip)
{
	netif_rx_fnc rx_fnc = in_interrupt() ? netif_rx : netif_rx_ni;
	struct pcpu_dstats *dstats;

	skb->protocol = ip ? htons(ETH_P_IP) : htons(ETH_P_ALL);
	skb->pkt_type = PACKET_HOST;

	netdev_dbg(ndev, "%s: skb=%p, len=%d c0=%d\n",
		   __func__, skb, skb->len, skb->data[0]);

	if (rx_fnc(skb) != NET_RX_SUCCESS) {
		ndev->stats.rx_dropped++;
		dev_kfree_skb(skb);
		return NET_RX_DROP;
	}

	ndev->stats.rx_bytes += skb->len;
	ndev->stats.rx_packets++;

	dstats = this_cpu_ptr(ndev->dstats);
	u64_stats_update_begin(&dstats->syncp);
	dstats->rx_bytes += skb->len;
	dstats->rx_packets++;
	u64_stats_update_end(&dstats->syncp);

	return NET_RX_SUCCESS;
}

int escape_string_buffer(char *buff,
			 unsigned int maxlen,
			 const unsigned char *data,
			 unsigned int len)
{
	static char *hexvals = "0123456789abcdef";
	int i;
	int out_len = 0;

	for (i = 0; (i < len) && (out_len < maxlen); i++) {
		unsigned char c = data[i];

		if (c >= 0x20 && c <= 0x7E) {
			if (buff)
				*(buff++) = c;
			out_len++;
		} else {
			if (buff) {
				*(buff++) = '\\';
				*(buff++) = 'x';
				*(buff++) = hexvals[c >> 4];
				*(buff++) = hexvals[c & 0xf];
			}
			out_len += 4;
		}
	}
	return out_len;
}

/**
 *	amazon_net_ndev_priv - access amazon network device private data
 *	@dev: network device
 *
 * Get network device private data
 */
void *amazon_net_ndev_priv(const struct net_device *ndev)
{
	return netdev_priv((!ndev->parent) ? ndev : ndev->parent);
}

module_init(amazon_net_init_module);
module_exit(amazon_net_cleanup_module);
MODULE_LICENSE("GPL");
MODULE_ALIAS_RTNL_LINK(DRV_NAME);
MODULE_VERSION(DRV_VERSION);
